const uxp = require('uxp');

// 简化的图层监听器
class SimpleLayerListener {
    constructor() {
        this.isListening = false;
        this.layerCount = 0;
        this.layerIds = new Set();
        this.init();
    }

    // 初始化
    async init() {
        console.log('图层监听器启动');
        setTimeout(() => {
            this.startListening();
        }, 500);
    }

    // 开始监听
    async startListening() {
        if (this.isListening) return;

        try {
            const app = require('photoshop').app;
            if (!app.activeDocument) {
                console.log('等待文档打开...');
                return;
            }

            this.isListening = true;
            console.log('开始监听图层变化');

            // 初始化状态
            await this.updateLayerState();

            // 打印当前图层
            this.printLayers();

            // 定时检查
            this.checkInterval = setInterval(() => {
                this.checkChanges();
            }, 500);

        } catch (error) {
            console.error('启动失败:', error);
            this.isListening = false;
        }
    }

    // 更新图层状态
    async updateLayerState() {
        try {
            const app = require('photoshop').app;
            const doc = app.activeDocument;

            if (!doc) return;

            this.layerCount = doc.layers.length;
            this.layerIds.clear();

            for (let i = 0; i < doc.layers.length; i++) {
                this.layerIds.add(doc.layers[i].id);
            }

        } catch (error) {
            console.error('更新状态失败:', error);
        }
    }

    // 检查变化
    async checkChanges() {
        try {
            const app = require('photoshop').app;
            const doc = app.activeDocument;

            if (!doc) {
                if (this.isListening) {
                    console.log('文档关闭，停止监听');
                    this.stopListening();
                }
                return;
            }

            const currentCount = doc.layers.length;

            if (currentCount !== this.layerCount) {
                console.log('检测到图层变化，当前图层:');
                this.printLayers();

                // 发送图层变化到webview
                this.sendLayerChangeToWebview();

                await this.updateLayerState();
            }

        } catch (error) {
            console.error('检查变化失败:', error);
        }
    }

    // 打印所有图层
    printLayers() {
        try {
            const app = require('photoshop').app;
            const doc = app.activeDocument;

            if (!doc) {
                console.log('没有活动文档');
                return;
            }

            console.log(`图层总数: ${doc.layers.length}`);

            for (let i = 0; i < doc.layers.length; i++) {
                const layer = doc.layers[i];
                console.log(`ID: ${layer.id}, Name: "${layer.name}"`);
            }
            console.log('---');

        } catch (error) {
            console.error('打印图层失败:', error);
        }
    }

    // 停止监听
    stopListening() {
        if (this.checkInterval) {
            clearInterval(this.checkInterval);
            this.checkInterval = null;
        }
        this.isListening = false;
        console.log('停止监听');
    }

    // 发送图层变化到webview
    sendLayerChangeToWebview() {
        // 使用全局的发送函数
        sendLayersToVue();
    }

    // 获取当前图层信息
    getCurrentLayersInfo() {
        try {
            const app = require('photoshop').app;
            const doc = app.activeDocument;

            if (!doc) {
                return [];
            }

            return doc.layers.map(layer => ({
                id: layer.id,
                name: layer.name,
                visible: layer.visible,
                opacity: layer.opacity
            }));
        } catch (error) {
            console.log('❌ 获取图层信息失败:', error.message);
            return [];
        }
    }

    /**
     * 上传当前文档的所有图层图像
     */
    async uploadCurrentDocument() {
        try {
            const app = require('photoshop').app;
            const doc = app.activeDocument;

            if (!doc) {
                console.log('❌ 没有活动文档');
                return;
            }

            console.log('🚀 开始上传图层图像数据...');
            await this.uploadLayersAsImages(doc);
            console.log('✅ 所有图层上传完成');

        } catch (error) {
            console.error('💥 上传失败:', error.message);
        }
    }

    /**
     * 上传所有图层作为图像
     * @param {Document} doc - Photoshop文档对象
     */
    async uploadLayersAsImages(doc) {
        try {
            console.log(`📋 处理 ${doc.layers.length} 个图层...`);

            for (let i = 0; i < doc.layers.length; i++) {
                const layer = doc.layers[i];

                try {
                    console.log(`🔄 处理图层 ${i + 1}: ${layer.name} (ID: ${layer.id})`);

                    const imageBlob = await this.createLayerImage(layer, i);
                    if (imageBlob) {
                        await this.uploadImageBlob(imageBlob, layer.name, layer.id);
                    } else {
                        console.log(`⚠️ 图层 ${layer.name} 无法创建图像`);
                    }

                } catch (layerError) {
                    console.error(`💥 处理图层 ${layer.name} 失败:`, layerError.message);
                }
            }

        } catch (error) {
            console.error('💥 上传图层图像失败:', error.message);
        }
    }

    /**
     * 为单个图层创建图像
     * @param {Layer} layer - Photoshop图层对象
     * @param {number} index - 图层索引
     * @returns {Blob|null} 图像Blob对象或null
     */
    async createLayerImage(layer, index) {
        try {
            const fs = require('uxp').storage.localFileSystem;
            const { core } = require('photoshop');
            const { batchPlay } = require('photoshop').action;
            const app = require('photoshop').app;

            console.log(`🖼️ 获取图层 ${layer.name} 的图像数据...`);

            return await core.executeAsModal(async () => {
                const activeDoc = app.activeDocument;
                const layers = activeDoc.layers;

                // 备份图层可见性状态
                const visibleState = layers.map(layerItem => layerItem.visible);

                // 记录原始历史状态
                const originalHistoryState = activeDoc.activeHistoryState;

                // 基于边界判断是否需要 trim（避免无透明区域时 trim 导致异常/多余操作）
                const canTrim = (() => {
                    try {
                        const b = layer.bounds;
                        // bounds 结构通常为 {left, top, right, bottom}
                        // 无法精确判断透明像素，但可以用“图层边界是否小于画布”作为启发式
                        return !!(activeDoc && b &&
                            (Number(b.left) > 0 || Number(b.top) > 0 ||
                                Number(b.right) < Number(activeDoc.width) ||
                                Number(b.bottom) < Number(activeDoc.height)));
                    } catch (_) {
                        return true;
                    }
                })();

                try {
                    // 临时隐藏除指定图层以外的所有图层
                    layers.forEach(layerItem => {
                        if (layerItem.id !== layer.id) {
                            layerItem.visible = false;
                        }
                    });

                    if (canTrim) {
                        console.log(`图层 ${layer.name} 可能存在透明区域，执行裁剪...`);
                        await batchPlay([
                            {
                                _obj: 'trim',
                                trimBasedOn: {
                                    _enum: 'trimBasedOn',
                                    _value: 'transparency'
                                },
                                top: true,
                                bottom: true,
                                left: true,
                                right: true,
                                _options: {
                                    dialogOptions: 'dontDisplay'
                                }
                            }
                        ], { synchronousExecution: true });
                    } else {
                        console.log(`图层 ${layer.name} 没有透明区域，跳过裁剪。`);
                    }

                    // 创建临时文件
                    const tempFolder = await fs.getTemporaryFolder();
                    const tempFile = await tempFolder.createFile(`layer_${layer.id}.jpg`, { overwrite: true });

                    // 导出为 JPG
                    await activeDoc.saveAs.jpg(tempFile, {
                        quality: 12,
                        embedColorProfile: true
                    });

                    // 读取文件内容
                    const arrayBuffer = await tempFile.read({ format: require('uxp').storage.formats.binary });
                    const uint8Array = new Uint8Array(arrayBuffer);

                    console.log(`📊 图层 ${layer.name} 导出成功: ${(uint8Array.length / 1024).toFixed(1)} KB`);

                    // 删除临时文件
                    try { await tempFile.delete(); } catch (_) {}

                    // 创建 Blob
                    const blob = new Blob([uint8Array], { type: 'image/jpeg' });
                    return blob;

                } finally {
                    // 恢复到原始历史状态（撤销裁剪操作）
                    try {
                        activeDoc.activeHistoryState = originalHistoryState;
                    } catch (restoreError) {
                        console.error('⚠️ 恢复历史状态失败:', restoreError);
                    }

                    // 恢复图层可见性
                    layers.forEach((layerItem, i) => {
                        layerItem.visible = visibleState[i];
                    });
                }

            }, { commandName: '导出图层图像' });

        } catch (error) {
            console.error(`💥 创建图层 ${layer.name} 图像失败:`, error);
            console.error('错误详情:', error.message || error);
            return null;
        }
    }

    /**
     * 将字符串转换为Uint8Array（UXP兼容方法）
     * @param {string} str - 要转换的字符串
     * @returns {Uint8Array} 字节数组
     */
    stringToUint8Array(str) {
        const bytes = new Uint8Array(str.length);
        for (let i = 0; i < str.length; i++) {
            bytes[i] = str.charCodeAt(i) & 0xFF;
        }
        return bytes;
    }

    /**
     * 构建multipart/form-data格式的HTTP请求体
     * @param {Uint8Array} imageData - 图像二进制数据
     * @param {string} fileName - 文件名
     * @returns {Object} 包含请求体和边界的对象
     */
    buildMultipartBody(imageData, fileName) {
        const boundary = '----WebKitFormBoundary' + Math.random().toString(16).substr(2);
        const CRLF = '\r\n';

        // 构建multipart头部
        const header = [
            `--${boundary}${CRLF}`,
            `Content-Disposition: form-data; name="image"; filename="${fileName}"${CRLF}`,
            `Content-Type: image/png${CRLF}`,
            CRLF
        ].join('');

        // 构建multipart尾部
        const footer = `${CRLF}--${boundary}--${CRLF}`;

        // 转换为字节数组
        const headerBytes = this.stringToUint8Array(header);
        const footerBytes = this.stringToUint8Array(footer);

        // 合并所有数据
        const totalLength = headerBytes.length + imageData.length + footerBytes.length;
        const combinedArray = new Uint8Array(totalLength);

        let offset = 0;
        combinedArray.set(headerBytes, offset);
        offset += headerBytes.length;
        combinedArray.set(imageData, offset);
        offset += imageData.length;
        combinedArray.set(footerBytes, offset);

        return {
            body: combinedArray,
            boundary: boundary,
            contentType: `multipart/form-data; boundary=${boundary}`
        };
    }

    /**
     * 上传选区图像（使用套索或矩形选区工具选择的图像）
     */
    async uploadSelection() {
        try {
            const app = require('photoshop').app;
            const activeDoc = app.activeDocument;

            if (!activeDoc) {
                console.log('❌ 没有活动文档');
                return;
            }

            // 检查是否有选区
            const selectionBounds = await this.getSelectionBounds();
            if (!selectionBounds) {
                console.log('❌ 没有选区，请先使用套索或矩形工具创建选区');
                return;
            }

            console.log('🚀 开始上传选区图像...');
            console.log(`📐 选区尺寸: ${selectionBounds.right - selectionBounds.left} x ${selectionBounds.bottom - selectionBounds.top}`);

            const imageBlob = await this.createSelectionImage();
            if (imageBlob) {
                const fileName = `selection_${Date.now()}.png`;
                await this.uploadImageBlob(imageBlob, fileName, 'selection');
                console.log('✅ 选区图像上传完成');
            } else {
                console.log('❌ 选区图像创建失败');
            }

        } catch (error) {
            console.error('💥 上传选区失败:', error.message);
        }
    }

    /**
     * 上传选中的图层
     */
    async uploadSelectedLayer() {
        try {
            const app = require('photoshop').app;
            const activeDoc = app.activeDocument;

            if (!activeDoc) {
                console.log('❌ 没有活动文档');
                return;
            }

            // 调试：显示所有图层信息
            console.log('📋 文档图层信息:');
            console.log(`总图层数: ${activeDoc.layers.length}`);
            console.log(`activeLayer:`, activeDoc.activeLayer);
            console.log(`activeLayers:`, activeDoc.activeLayers);

            // 尝试多种方式获取选中图层
            let activeLayer = null;

            // 方法1：使用activeLayer
            if (activeDoc.activeLayer) {
                activeLayer = activeDoc.activeLayer;
                console.log('✅ 通过activeLayer获取到图层');
            }
            // 方法2：使用activeLayers数组
            else if (activeDoc.activeLayers && activeDoc.activeLayers.length > 0) {
                activeLayer = activeDoc.activeLayers[0];
                console.log('✅ 通过activeLayers获取到图层');
            }
            // 方法3：使用第一个可见图层作为备选
            else {
                const visibleLayers = activeDoc.layers.filter(layer => layer.visible);
                if (visibleLayers.length > 0) {
                    activeLayer = visibleLayers[0];
                    console.log('⚠️ 使用第一个可见图层作为备选');
                }
            }

            if (!activeLayer) {
                console.log('❌ 没有找到可用的图层');
                console.log('💡 请确保至少有一个图层存在且可见');
                return;
            }

            console.log('🚀 开始上传选中图层...');
            console.log(`📋 选中图层: ${activeLayer.name} (ID: ${activeLayer.id})`);
            console.log(`👁️ 图层可见性: ${activeLayer.visible}`);

            const imageBlob = await this.createLayerImage(activeLayer, 0);
            if (imageBlob) {
                await this.uploadImageBlob(imageBlob, activeLayer.name, activeLayer.id);
                console.log('✅ 选中图层上传完成');
            } else {
                console.log('❌ 选中图层图像创建失败');
            }

        } catch (error) {
            console.error('💥 上传选中图层失败:', error.message);
        }
    }

    /**
     * 获取选区边界信息
     * @returns {Object|null} 选区边界对象或null
     */
    async getSelectionBounds() {
        try {
            const app = require('photoshop').app;
            const selection = app.activeDocument.selection;
            const selectionBounds = selection.bounds;

            if (selectionBounds) {
                // 检查是否为有效选区
                if (selectionBounds.left === 0 && selectionBounds.top === 0 &&
                    selectionBounds.right === 0 && selectionBounds.bottom === 0) {
                    return null;
                } else {
                    return {
                        left: selectionBounds.left,
                        top: selectionBounds.top,
                        right: selectionBounds.right,
                        bottom: selectionBounds.bottom
                    };
                }
            }
            return null;
        } catch (error) {
            console.error('获取选区边界失败:', error);
            return null;
        }
    }

    /**
     * 获取选区图像并转换为base64（用于webview显示）
     * @returns {string|null} base64编码的图像数据或null
     */
    async getSelectionImageBase64() {
        try {
            const fs = require('uxp').storage.localFileSystem;
            const {core} = require("photoshop");
            const {batchPlay} = require("photoshop").action;
            const app = require('photoshop').app;

            console.log('🖼️ 获取选区图像base64数据...');

            return await core.executeAsModal(async () => {
                const activeDoc = app.activeDocument;

                // 获取选区边界
                const selectionBounds = await this.getSelectionBounds();
                if (!selectionBounds) {
                    throw new Error('没有选区或选区无效');
                }

                console.log('选区边界:', selectionBounds);

                // 复制选区到剪贴板
                await batchPlay([
                    {
                        _obj: "copyEvent",
                        copyHint: "pixels",
                        _options: {
                            dialogOptions: "dontDisplay"
                        }
                    }
                ], {
                    synchronousExecution: true
                });

                console.log('选区已复制到剪贴板');

                // 创建新文档
                await batchPlay([
                    {
                        _obj: "make",
                        new: {
                            _obj: "document",
                            artboard: false,
                            autoPromoteBackgroundLayer: false,
                            preset: "剪贴板"
                        },
                        _options: {
                            dialogOptions: "dontDisplay"
                        }
                    }
                ], {
                    synchronousExecution: true
                });

                console.log('新文档创建成功');

                // 粘贴选区内容
                await batchPlay([
                    {
                        _obj: "paste",
                        antiAlias: {
                            _enum: "antiAliasType",
                            _value: "antiAliasNone"
                        },
                        as: {
                            _class: "pixel"
                        },
                        _options: {
                            dialogOptions: "dontDisplay"
                        }
                    }
                ], {
                    synchronousExecution: true
                });

                console.log('选区内容已粘贴');

                // 创建临时文件
                const tempFolder = await fs.getTemporaryFolder();
                const tempFile = await tempFolder.createFile(`selection_${Date.now()}.png`, {overwrite: true});

                // 保存新文档为PNG
                await app.activeDocument.saveAs.png(tempFile, {
                    quality: 1,
                    interlaced: false,
                    transparency: true
                });

                console.log('PNG导出完成');

                // 读取文件内容
                const arrayBuffer = await tempFile.read({format: require('uxp').storage.formats.binary});
                const uint8Array = new Uint8Array(arrayBuffer);

                console.log(`📊 选区图像导出成功: ${(uint8Array.length / 1024).toFixed(1)} KB`);

                // 关闭新文档
                await app.activeDocument.close();

                // 转换为base64
                const base64 = this.uint8ToBase64(uint8Array);
                return base64;

            }, {commandName: "导出选区图像为base64"});

        } catch (error) {
            console.error(`💥 获取选区图像base64失败:`, error);
            console.error('错误详情:', error.message || error);
            return null;
        }
    }

    /**
     * 将Uint8Array转换为base64字符串
     * @param {Uint8Array} uint8Array - 字节数组
     * @returns {string} base64字符串
     */
    uint8ToBase64(uint8Array) {
        const CHUNK_SIZE = 0x8000;
        let index = 0;
        const length = uint8Array.length;
        let result = '';
        while (index < length) {
            const slice = uint8Array.subarray(index, Math.min(index + CHUNK_SIZE, length));
            result += String.fromCharCode(...slice);
            index += CHUNK_SIZE;
        }
        return btoa(result);
    }

    /**
     * 创建选区图像（基于参考代码的改进版本）
     * @returns {Blob|null} 选区图像Blob对象或null
     */
    async createSelectionImage() {
        try {
            const fs = require('uxp').storage.localFileSystem;
            const {core} = require("photoshop");
            const {batchPlay} = require("photoshop").action;
            const app = require('photoshop').app;

            console.log('🖼️ 获取选区图像数据...');

            return await core.executeAsModal(async () => {
                const activeDoc = app.activeDocument;

                // 获取选区边界
                const selectionBounds = await this.getSelectionBounds();
                if (!selectionBounds) {
                    throw new Error('没有选区或选区无效');
                }

                console.log('选区边界:', selectionBounds);

                // 复制选区到剪贴板
                await batchPlay([
                    {
                        _obj: "copyEvent",
                        copyHint: "pixels",
                        _options: {
                            dialogOptions: "dontDisplay"
                        }
                    }
                ], {
                    synchronousExecution: true
                });

                console.log('选区已复制到剪贴板');

                // 创建新文档
                await batchPlay([
                    {
                        _obj: "make",
                        new: {
                            _obj: "document",
                            artboard: false,
                            autoPromoteBackgroundLayer: false,
                            preset: "剪贴板"
                        },
                        _options: {
                            dialogOptions: "dontDisplay"
                        }
                    }
                ], {
                    synchronousExecution: true
                });

                console.log('新文档创建成功');

                // 粘贴选区内容
                await batchPlay([
                    {
                        _obj: "paste",
                        antiAlias: {
                            _enum: "antiAliasType",
                            _value: "antiAliasNone"
                        },
                        as: {
                            _class: "pixel"
                        },
                        _options: {
                            dialogOptions: "dontDisplay"
                        }
                    }
                ], {
                    synchronousExecution: true
                });

                console.log('选区内容已粘贴');

                // 创建临时文件
                const tempFolder = await fs.getTemporaryFolder();
                const tempFile = await tempFolder.createFile(`selection_${Date.now()}.png`, {overwrite: true});

                // 保存新文档为PNG
                await app.activeDocument.saveAs.png(tempFile, {
                    quality: 1,
                    interlaced: false,
                    transparency: true
                });

                console.log('PNG导出完成');

                // 读取文件内容
                const arrayBuffer = await tempFile.read({format: require('uxp').storage.formats.binary});
                const uint8Array = new Uint8Array(arrayBuffer);

                console.log(`📊 选区图像导出成功: ${(uint8Array.length / 1024).toFixed(1)} KB`);

                // 关闭新文档
                await app.activeDocument.close();

                // 创建Blob
                const blob = new Blob([uint8Array], {type: 'image/png'});
                return blob;

            }, {commandName: "导出选区图像"});

        } catch (error) {
            console.error(`💥 创建选区图像失败:`, error);
            console.error('错误详情:', error.message || error);
            return null;
        }
    }

    /**
     * 上传图像Blob到服务器
     * @param {Blob} imageBlob - 图像Blob对象
     * @param {string} layerName - 图层名称
     * @param {string} layerId - 图层ID
     */
    async uploadImageBlob(imageBlob, layerName, layerId) {
        try {
            // 生成文件名
            const cleanLayerName = layerName.replace(/[^a-zA-Z0-9\u4e00-\u9fa5]/g, '_');
            const fileName = `${cleanLayerName}_${layerId}.png`;

            console.log(`📤 上传图层: ${layerName} (${layerId})`);
            console.log(`📁 文件名: ${fileName}`);
            console.log(`📊 文件大小: ${(imageBlob.size / 1024).toFixed(1)} KB`);

            // 获取图像数据
            const arrayBuffer = await imageBlob.arrayBuffer();
            const imageData = new Uint8Array(arrayBuffer);

            // 构建multipart请求体
            const multipartData = this.buildMultipartBody(imageData, fileName);

            console.log(`🔗 请求体大小: ${(multipartData.body.length / 1024).toFixed(1)} KB`);

            // 发送上传请求
            const uploadUrl = 'http://192.168.1.4:8188/upload/image';
            const response = await fetch(uploadUrl, {
                method: 'POST',
                headers: {
                    'Accept': '*/*',
                    'Cache-Control': 'no-cache',
                    'User-Agent': 'PS-UXP-Plugin/1.0.0',
                    'Content-Type': multipartData.contentType,
                    'Content-Length': multipartData.body.length.toString()
                },
                body: multipartData.body
            });

            // 处理响应
            if (response.ok) {
                const result = await response.text();
                console.log(`✅ 图层 ${layerName} 上传成功!`);
                console.log(`📝 服务器响应: ${result}`);
            } else {
                console.error(`❌ 图层 ${layerName} 上传失败: ${response.status} ${response.statusText}`);
                const errorText = await response.text();
                console.error(`🔍 错误详情: ${errorText}`);
            }

        } catch (error) {
            console.error(`💥 上传图层 ${layerName} 时发生错误:`, error.message);
        }
    }
}

// 创建监听器实例
const layerListener = new SimpleLayerListener();

// ============= 存储功能 =============
// 存储数据
function storageSet(key, value) {
    try {
        localStorage.setItem(key, value);
        console.log('✅ 存储成功:', key);
        return true;
    } catch (error) {
        console.error('❌ 存储失败:', key, error);
        return false;
    }
}

// 获取数据
function storageGet(key) {
    try {
        const value = localStorage.getItem(key);
        if (value !== null) {
            console.log('✅ 读取成功:', key);
            return value;
        }
        console.log('⚠️ 数据不存在:', key);
        return null;
    } catch (error) {
        console.error('❌ 读取失败:', key, error);
        return null;
    }
}

// 删除数据
function storageRemove(key) {
    try {
        localStorage.removeItem(key);
        console.log('✅ 删除成功:', key);
        return true;
    } catch (error) {
        console.error('❌ 删除失败:', key, error);
        return false;
    }
}

// 清空所有数据
function storageClear() {
    try {
        localStorage.clear();
        console.log('✅ 清空存储成功');
        return true;
    } catch (error) {
        console.error('❌ 清空存储失败:', error);
        return false;
    }
}

// 监听来自webview的消息
window.addEventListener("message", async (e) => {
    console.log('📨 收到webview消息:', e);
    console.log('📍 消息来源:', e.origin);
    console.log('📦 消息数据:', e.data);

    // 检查是否是来自Vue项目的消息
    if (e.origin === 'http://localhost:3000') {
        console.log('🎯 确认来自Vue项目的消息');
    }

    const data = (() => {
        try {
            return typeof e.data === 'string' ? JSON.parse(e.data) : e.data;
        } catch (err) {
            console.log('⚠️ 消息解析失败(非JSON字符串)，按原样处理:', e.data);
            return e.data;
        }
    })();

    console.log('🔍 消息类型:', data && data.action);
    if (data && data.action === 'imageDisplayed') {
        console.log('✅ webview确认图像已显示');
    } else if (data && data.action === 'layerChangeReceived') {
        console.log('✅ webview确认图层变化已接收');
    } else if (data && data.action === 'vueMessage') {
        console.log('🎉 收到Vue项目消息:', data.message);
        console.log('⏰ 时间戳:', data.timestamp);
    } else if (data && data.action === 'getLayers') {
        console.log('📋 Vue请求获取图层信息');
        // 立即发送当前图层信息
        sendLayersToVue();
    } else if (data && data.action === 'getLayerImage') {
        console.log('🖼️ Vue请求获取指定图层图像:', data.layerId, 'field:', data.field);
        // 获取指定图层的图像（layerId 可能是字符串，这里强制转 number，避免找不到图层）
        getLayerImageAndSend(Number(data.layerId), data.layerName, data.field);
    } else if (data && data.action === 'getDocumentImage') {
        console.log('📄 Vue请求获取当前文档图像:', data.field);

        // 检查是否有选区
        const selectionBounds = await layerListener.getSelectionBounds();
        console.log('检查是否有选区', selectionBounds);

        if (!selectionBounds) {
            // 获取当前文档的图像
            await getDocumentImageAndSend(data.field);
        } else {
            // 获取选区图像和去掉选区后的文档图像
            await getSelectionAndDocumentAndSend(data.field);
        }
        // 获取当前文档的图像
        // getDocumentImageAndSend(data.field);
    } else if (data && data.action === 'getSelectionAndDocument') {
        console.log('✂️ Vue请求获取选区和文档图像:', data.field);
        // 获取选区图像和去掉选区后的文档图像
        getSelectionAndDocumentAndSend(data.field);
    } else if (data && data.action === 'getSelectionImage') {
        console.log('✂️ Vue请求获取选区图像:', data.field);
        // 获取选区图像
        getSelectionImageAndSend(data.field);
    } else if (data && data.action === 'addToNewDocument') {
        console.log('📄 Vue请求添加到新文档:', data.imageUrl);
        // 将图像添加到新文档
        addImageToNewDocument(data.imageUrl, data.timestamp);
    } else if (data && data.action === 'addToSelection') {
        console.log('✂️ Vue请求添加到选区:', data.imageUrl);
        // 将图像添加到当前选区
        addImageToSelection(data.imageUrl, data.timestamp);
    } else if (data && data.action === 'executeAction') {
        console.log('⚡ Vue请求执行动作:', data.actionName);
        // 执行Photoshop动作
        executeAction(data.actionData, data.actionName, data.timestamp);
    } else if (data && data.action === 'storageSet') {
        console.log('💾 Vue请求存储数据:', data.key);
        // 存储数据
        const success = storageSet(data.key, data.value);
        const webview = document.getElementById('panelWebview');
        if (webview) {
            webview.postMessage({
                action: 'storageSetResponse',
                requestId: data.requestId,
                success: success
            });
        }
    } else if (data && data.action === 'storageGet') {
        console.log('📖 Vue请求读取数据:', data.key);
        // 读取数据
        const value = storageGet(data.key);
        const webview = document.getElementById('panelWebview');
        if (webview) {
            webview.postMessage({
                action: 'storageGetResponse',
                requestId: data.requestId,
                value: value
            });
        }
    } else if (data && data.action === 'storageRemove') {
        console.log('🗑️ Vue请求删除数据:', data.key);
        // 删除数据
        const success = storageRemove(data.key);
        const webview = document.getElementById('panelWebview');
        if (webview) {
            webview.postMessage({
                action: 'storageRemoveResponse',
                requestId: data.requestId,
                success: success
            });
        }
    } else if (data && data.action === 'storageClear') {
        console.log('🧹 Vue请求清空存储');
        // 清空存储
        const success = storageClear();
        const webview = document.getElementById('panelWebview');
        if (webview) {
            webview.postMessage({
                action: 'storageClearResponse',
                requestId: data.requestId,
                success: success
            });
        }
    } else if (data && data.action === 'saveAllHistoryImages') {
        console.log('🖼️ Vue请求保存全部历史图片, 数量:', (data.items || []).length);
        await saveAllHistoryImagesFromVue(data);
    } else if (data && data.action === 'saveTextFile') {
        console.log('💾 Vue请求保存文本文件:', data.filename);
        // 保存文本内容到本地文件（会弹出保存对话框）
        await saveTextFileFromVue(data);
    } else {
        console.log('📨 收到其他消息:', data);
    }
});


// 发送图层数据到Vue项目
function sendLayersToVue() {
    try {
        const webview = document.getElementById('panelWebview');
        if (webview) {
            const app = require('photoshop').app;
            const doc = app.activeDocument;

            if (!doc) {
                console.log('❌ 没有活动文档');
                // 发送空图层信息
                webview.postMessage({
                    action: 'layer',
                    timestamp: new Date().toLocaleString(),
                    layers: [],
                    error: '没有活动文档'
                });
                return;
            }

            const layers = doc.layers.map(layer => ({
                id: layer.id,
                name: layer.name,
                visible: layer.visible,
                opacity: layer.opacity,
                bounds: layer.bounds, // 图层边界
                kind: layer.kind,    // 图层类型
                selected: layer.selected // 是否选中
            }));

            const message = {
                action: 'layer',
                timestamp: new Date().toLocaleString(),
                layers: layers,
                documentName: doc.name,
                layerCount: layers.length
            };

            webview.postMessage(message);
            console.log('📤 图层数据已发送到Vue:', {
                documentName: doc.name,
                layerCount: layers.length,
                layers: layers.map(l => `${l.id}:${l.name}`)
            });
        } else {
            console.log('⚠️ 未找到webview元素');
        }
    } catch (error) {
        console.log('❌ 发送图层数据到Vue失败:', error.message);
        // 发送错误信息
        const webview = document.getElementById('panelWebview');
        if (webview) {
            webview.postMessage({
                action: 'layer',
                timestamp: new Date().toLocaleString(),
                layers: [],
                error: error.message
            });
        }
    }
}

// 获取指定图层的图像并发送到Vue
async function getLayerImageAndSend(layerId, layerName, field) {
    try {
        const webview = document.getElementById('panelWebview');
        if (!webview) {
            console.log('⚠️ 未找到webview元素');
            return;
        }

        const app = require('photoshop').app;
        const doc = app.activeDocument;

        if (!doc) {
            console.log('❌ 没有活动文档');
            webview.postMessage({
                action: 'layerImage',
                layerId: layerId,
                layerName: layerName,
                field: field,
                timestamp: new Date().toLocaleString(),
                error: '没有活动文档'
            });
            return;
        }

        // 查找指定图层
        const targetLayer = doc.layers.find(layer => layer.id === layerId);
        if (!targetLayer) {
            console.log('❌ 未找到指定图层:', layerId);
            webview.postMessage({
                action: 'layerImage',
                layerId: layerId,
                layerName: layerName,
                field: field,
                timestamp: new Date().toLocaleString(),
                error: '未找到指定图层'
            });
            return;
        }

        console.log('🎯 开始获取图层图像:', targetLayer.name);

        // 获取图层图像
        const imageBlob = await layerListener.createLayerImage(targetLayer, layerId);

        if (imageBlob) {
            // 转换为base64
            const arrayBuffer = await imageBlob.arrayBuffer();
            const uint8Array = new Uint8Array(arrayBuffer);
            const base64 = layerListener.uint8ToBase64(uint8Array);

            // 发送图像数据到Vue
            webview.postMessage({
                action: 'layerImage',
                layerId: layerId,
                layerName: layerName || targetLayer.name,
                field: field,
                timestamp: new Date().toLocaleString(),
                base64: base64,
                imageSize: `${targetLayer.bounds.right - targetLayer.bounds.left}x${targetLayer.bounds.bottom - targetLayer.bounds.top}`,
                dataSize: Math.round(base64.length * 0.75 / 1024) + 'KB'
            });

            console.log('📤 图层图像已发送到Vue:', {
                layerId: layerId,
                layerName: targetLayer.name,
                field: field,
                imageSize: `${targetLayer.bounds.right - targetLayer.bounds.left}x${targetLayer.bounds.bottom - targetLayer.bounds.top}`,
                dataSize: Math.round(base64.length * 0.75 / 1024) + 'KB'
            });
        } else {
            console.log('❌ 图层图像获取失败');
            webview.postMessage({
                action: 'layerImage',
                layerId: layerId,
                layerName: layerName || targetLayer.name,
                field: field,
                timestamp: new Date().toLocaleString(),
                error: '图层图像获取失败'
            });
        }

    } catch (error) {
        console.log('❌ 获取图层图像失败:', error.message);
        const webview = document.getElementById('panelWebview');
        if (webview) {
            webview.postMessage({
                action: 'layerImage',
                layerId: layerId,
                layerName: layerName,
                field: field,
                timestamp: new Date().toLocaleString(),
                error: error.message
            });
        }
    }
}

// 导出整个文档为PNG
async function exportDocumentAsPNG(doc) {
    try {
        const fs = require('uxp').storage.localFileSystem;
        const {core} = require("photoshop");

        return await core.executeAsModal(async () => {
            // 创建临时文件
            const tempFolder = await fs.getTemporaryFolder();
            const tempFile = await tempFolder.createFile(`document_${Date.now()}.png`, {overwrite: true});

            // 导出整个文档为PNG
            await doc.saveAs.png(tempFile, {
                quality: 1,
                interlaced: false,
                transparency: true
            });

            // 读取文件内容
            const arrayBuffer = await tempFile.read({format: require('uxp').storage.formats.binary});
            const uint8Array = new Uint8Array(arrayBuffer);

            console.log(`📊 文档 ${doc.name} 导出成功: ${(uint8Array.length / 1024).toFixed(1)} KB`);

            // 创建Blob
            const blob = new Blob([uint8Array], {type: 'image/png'});

            // 删除临时文件
            await tempFile.delete();

            return blob;

        }, {commandName: "导出文档图像"});

    } catch (error) {
        console.error(`💥 导出文档 ${doc.name} 失败:`, error.message);
        return null;
    }
}

// 获取当前文档的图像并发送到Vue
async function getDocumentImageAndSend(field) {
    try {
        const webview = document.getElementById('panelWebview');
        if (!webview) {
            console.log('⚠️ 未找到webview元素');
            return;
        }

        const app = require('photoshop').app;
        const doc = app.activeDocument;

        if (!doc) {
            console.log('❌ 没有活动文档');
            webview.postMessage({
                action: 'documentImage',
                field: field,
                timestamp: new Date().toLocaleString(),
                error: '没有活动文档'
            });
            return;
        }

        console.log('🎯 开始获取文档图像:', doc.name);

        // 获取文档图像
        const imageBlob = await exportDocumentAsPNG(doc);

        if (imageBlob) {
            // 转换为base64
            const arrayBuffer = await imageBlob.arrayBuffer();
            const uint8Array = new Uint8Array(arrayBuffer);
            const base64 = layerListener.uint8ToBase64(uint8Array);

            // 发送图像数据到Vue
            webview.postMessage({
                action: 'documentImage',
                field: field,
                timestamp: new Date().toLocaleString(),
                base64: base64,
                documentName: doc.name,
                imageSize: `${doc.width}x${doc.height}`,
                dataSize: Math.round(base64.length * 0.75 / 1024) + 'KB'
            });

            console.log('📤 文档图像已发送到Vue:', {
                documentName: doc.name,
                field: field,
                imageSize: `${doc.width}x${doc.height}`,
                dataSize: Math.round(base64.length * 0.75 / 1024) + 'KB'
            });
        } else {
            console.log('❌ 文档图像获取失败');
            webview.postMessage({
                action: 'documentImage',
                field: field,
                timestamp: new Date().toLocaleString(),
                error: '文档图像获取失败'
            });
        }

    } catch (error) {
        console.log('❌ 获取文档图像失败:', error.message);
        const webview = document.getElementById('panelWebview');
        if (webview) {
            webview.postMessage({
                action: 'documentImage',
                field: field,
                timestamp: new Date().toLocaleString(),
                error: error.message
            });
        }
    }
}
// 获取选区变透明后的文档图像
async function createDocumentAfterSelectionDelete() {
    try {
        const fs = require('uxp').storage.localFileSystem;
        const { core } = require("photoshop");
        const { batchPlay } = require("photoshop").action;
        const app = require('photoshop').app;

        console.log('🖼️ 选择反向、拷贝图层、导出...');

        return await core.executeAsModal(async () => {
            const activeDoc = app.activeDocument;
            const originalHistoryState = activeDoc.activeHistoryState;

            try {
                // 判断是否有多个可见图层（不含组，递归统计）
                const countVisibleArtLayers = (layers) => {
                    let n = 0;
                    for (const layer of layers) {
                        if (!layer.visible) continue;
                        if (layer.typename === "LayerGroup") n += countVisibleArtLayers(layer.layers);
                        else n += 1;
                    }
                    return n;
                };

                const visibleCount = countVisibleArtLayers(activeDoc.layers);

                // 只有可见图层 >= 2 时才有必要 mergeVisible
                if (visibleCount >= 2) {
                    await batchPlay([
                        {
                            _obj: "mergeVisible",
                            duplicate: true,
                            _options: {
                               dialogOptions: "dontDisplay"
                            }
                         }
                    ], { synchronousExecution: true });
                } else {
                    console.log(`⚠️ 可见图层数量为 ${visibleCount}，跳过 mergeVisible`);
                }
                /* ===============================
                 * 1️⃣ 选择反向
                 * =============================== */
                await batchPlay([
                    
                    {
                        _obj: "inverse",
                        _options: { dialogOptions: "dontDisplay" }
                    }
                ], { synchronousExecution: true });

                console.log('✅ 已反选选区');

                /* ===============================
                 * 2️⃣ 拷贝图层（通过选区复制）
                 * =============================== */
                await batchPlay([
                    {
                        _obj: "copyToLayer",
                        _options: { dialogOptions: "dontDisplay" }
                    }
                ], { synchronousExecution: true });

                console.log('✅ 已拷贝图层');

                // 获取新创建的图层（应该是当前活动图层）
                const copiedLayer = activeDoc.activeLayers[0];
                console.log('新图层:', copiedLayer.name);

                // 隐藏所有其他图层，只显示拷贝的图层
                const layers = activeDoc.layers;
                layers.forEach(layer => {
                    layer.visible = (layer.id === copiedLayer.id);
                });

                /* ===============================
                 * 3️⃣ 导出拷贝的图层
                 * =============================== */
                const tempFolder = await fs.getTemporaryFolder();
                const tempFile = await tempFolder.createFile(
                    `copied_layer_${Date.now()}.png`,
                    { overwrite: true }
                );

                await activeDoc.saveAs.png(tempFile, {
                    quality: 1,
                    interlaced: false,
                    transparency: true  // 保留透明通道
                });

                // 读取文件内容
                const arrayBuffer = await tempFile.read({
                    format: require('uxp').storage.formats.binary
                });
                const uint8Array = new Uint8Array(arrayBuffer);

                console.log(`📊 导出成功: ${(uint8Array.length / 1024).toFixed(1)} KB`);

                const blob = new Blob([uint8Array], { type: 'image/png' });

                await tempFile.delete();

                return blob;

            } catch (error) {
                console.error(`💥 导出过程出错:`, error);
                throw error;
            } finally {
                // 恢复到原始历史状态
                try {
                    activeDoc.activeHistoryState = originalHistoryState;
                    
                    // 执行一个空操作来"固化"当前状态，清除重做历史
                    // 这会使灰色的历史记录消失
                    try {
                        await batchPlay([
                            {
                                _obj: "select",
                                _target: [{ _ref: "layer", _enum: "ordinal", _value: "targetEnum" }],
                                makeVisible: false,
                                _options: { dialogOptions: "dontDisplay" }
                            }
                        ], { synchronousExecution: true });
                    } catch (e) {
                        // 忽略错误，这只是为了清除重做历史
                    }
                    
                    console.log('🔙 文档状态已恢复');
                } catch (restoreError) {
                    console.error(`⚠️ 恢复历史状态失败:`, restoreError);
                }
            }

        }, { commandName: "导出选区外内容" });

    } catch (error) {
        console.error(`[object Object]`, error.message || error);
        return null;
    }
}



// 获取选区图像和去掉选区后的文档图像并发送到Vue
async function getSelectionAndDocumentAndSend(field) {
    try {
        const webview = document.getElementById('panelWebview');
        if (!webview) {
            console.log('⚠️ 未找到webview元素');
            return;
        }

        const app = require('photoshop').app;
        const doc = app.activeDocument;

        if (!doc) {
            console.log('❌ 没有活动文档');
            webview.postMessage({
                action: 'selectionAndDocument',
                field: field,
                timestamp: new Date().toLocaleString(),
                error: '没有活动文档'
            });
            return;
        }

        console.log('🎯 开始获取当前文档图像和选区删除后的文档图像');


        // 再获取选区删除后的文档图像
        const documentAfterDeleteBlob = await createDocumentAfterSelectionDelete();

        const result = {
            action: 'selectionAndDocument',
            field: field,
            timestamp: new Date().toLocaleString(),
            documentName: doc.name,
            imageSize: `${doc.width}x${doc.height}`
        };

        // 处理选区删除后的文档图像（作为documentImage）
        if (documentAfterDeleteBlob) {
            const afterDeleteArrayBuffer = await documentAfterDeleteBlob.arrayBuffer();
            const afterDeleteUint8Array = new Uint8Array(afterDeleteArrayBuffer);
            const afterDeleteBase64 = layerListener.uint8ToBase64(afterDeleteUint8Array);

            result.documentImage = {
                base64: afterDeleteBase64,
                dataSize: Math.round(afterDeleteBase64.length * 0.75 / 1024) + 'KB'
            };

            console.log('✅ 选区删除后的文档图像获取成功');
        } else {
            result.documentImage = {
                error: '选区删除后的文档图像获取失败'
            };
            console.log('❌ 选区删除后的文档图像获取失败');
        }

        // 发送结果到Vue
        webview.postMessage(result);

        console.log('📤 选区和文档图像已发送到Vue:', {
            documentName: doc.name,
            field: field,
            hasSelection: !!(result.selectionImage && result.selectionImage.base64),
            hasDocument: !!(result.documentImage && result.documentImage.base64)
        });

    } catch (error) {
        console.log('❌ 获取选区和文档图像失败:', error.message);
        const webview = document.getElementById('panelWebview');
        if (webview) {
            webview.postMessage({
                action: 'selectionAndDocument',
                field: field,
                timestamp: new Date().toLocaleString(),
                error: error.message
            });
        }
    }
}

// 获取选区图像并发送到Vue
async function getSelectionImageAndSend(field) {
    try {
        const webview = document.getElementById('panelWebview');
        if (!webview) {
            console.log('⚠️ 未找到webview元素');
            return;
        }

        const app = require('photoshop').app;
        const doc = app.activeDocument;

        if (!doc) {
            console.log('❌ 没有活动文档');
            webview.postMessage({
                action: 'selectionImage',
                field: field,
                timestamp: new Date().toLocaleString(),
                error: '没有活动文档'
            });
            return;
        }

        console.log('🎯 开始获取选区图像');

        // 获取选区图像
        const imageBlob = await layerListener.createSelectionImage();

        if (imageBlob) {
            // 转换为base64
            const arrayBuffer = await imageBlob.arrayBuffer();
            const uint8Array = new Uint8Array(arrayBuffer);
            const base64 = layerListener.uint8ToBase64(uint8Array);

            // 发送图像数据到Vue
            webview.postMessage({
                action: 'selectionImage',
                field: field,
                timestamp: new Date().toLocaleString(),
                base64: base64,
                documentName: doc.name,
                dataSize: Math.round(base64.length * 0.75 / 1024) + 'KB'
            });

            console.log('📤 选区图像已发送到Vue:', {
                documentName: doc.name,
                field: field,
                dataSize: Math.round(base64.length * 0.75 / 1024) + 'KB'
            });
        } else {
            console.log('❌ 选区图像获取失败');
            webview.postMessage({
                action: 'selectionImage',
                field: field,
                timestamp: new Date().toLocaleString(),
                error: '选区图像获取失败'
            });
        }

    } catch (error) {
        console.log('❌ 获取选区图像失败:', error.message);
        const webview = document.getElementById('panelWebview');
        if (webview) {
            webview.postMessage({
                action: 'selectionImage',
                field: field,
                timestamp: new Date().toLocaleString(),
                error: error.message
            });
        }
    }
}

// 将图像添加到新文档
async function addImageToNewDocument(imageUrl, timestamp) {
    try {
        console.log('📄 开始创建新文档并添加图像:', imageUrl);

        const {batchPlay} = require("photoshop").action;
        const {executeAsModal} = require("photoshop").core;
        const fs = require('uxp').storage.localFileSystem;

        // 获取临时文件夹
        const tempFolder = await fs.getTemporaryFolder();

        // 从 URL 中获取文件名
        const fileName = imageUrl.split('/').pop().split('?')[0];

        // 获取后缀名（包含 .）
        const ext = fileName.substring(fileName.lastIndexOf('.'));

        // 生成随机文件名（10 位随机字符 + 原后缀）
        const randomFileName = Math.random().toString(36).slice(2, 12) + ext;

        console.log("📁 随机文件名:", randomFileName);


        // 检查文件是否已存在
        let tempFile;
        // 文件不存在，下载并保存
        console.log('📥 开始下载图像...');
        const response = await fetch(imageUrl);
        if (!response.ok) throw new Error(`获取图片失败: ${response.statusText}`);

        const arrayBuffer = await response.arrayBuffer();
        tempFile = await tempFolder.createFile(randomFileName, {overwrite: true});
        await tempFile.write(arrayBuffer);
        console.log(`✅ 已下载并保存文件: ${randomFileName}`);

        // 创建资源令牌
        const assetToken = await fs.createSessionToken(tempFile);

        await executeAsModal(async () => {
            await batchPlay([
                {
                    _obj: "open",
                    null: {
                        _path: assetToken,
                        _kind: "local"
                    },
                    _options: {
                        dialogOptions: "dontDisplay"
                    }
                }
            ], {synchronousExecution: true});

            console.log('✅ 图像已添加到新文档');
        }, {commandName: "添加图像到新文档"});

    } catch (error) {
        console.log('❌ 添加图像到新文档失败:', error.message);
    }
}

// 将图像置入到当前文档（不依赖选区边界；按“文档坐标”对齐，避免视图缩放导致位置/尺寸错乱）
async function addImageToSelection(imageUrl, timestamp) {
    try {
        console.log('✂️ 开始将图像添加到选区:', imageUrl);

        const { batchPlay } = require("photoshop").action;
        const { executeAsModal } = require("photoshop").core;
        const { app } = require("photoshop");
        const fs = require('uxp').storage.localFileSystem;

        // 获取临时文件夹
        const tempFolder = await fs.getTemporaryFolder();

        // 从URL中提取文件名
        const url = new URL(imageUrl);
        const fileName = url.searchParams.get("filename") || "selection_image.png";

        console.log('📁 文件名:', fileName);

        // 直接下载并保存（不判断是否已存在）
            console.log('📥 开始下载图像...');
            const response = await fetch(imageUrl);
            if (!response.ok) throw new Error(`获取图片失败: ${response.statusText}`);

            const arrayBuffer = await response.arrayBuffer();
        const tempFile = await tempFolder.createFile(fileName, { overwrite: true });
            await tempFile.write(arrayBuffer);
            console.log(tempFile);
            console.log(`✅ 已下载并保存文件: ${fileName}`);

        // 创建资源令牌
        const assetToken = await fs.createSessionToken(tempFile);

        await executeAsModal(async () => {
            const doc = app.activeDocument;
            if (!doc) throw new Error('没有活动文档');

            // 1) 按你的方法：先选中画布（全选），再直接置入
            await batchPlay(
                [
                    {
                        _obj: "set",
                        _target: [{ _ref: "channel", _property: "selection" }],
                        to: { _enum: "ordinal", _value: "allEnum" },
                        _options: { dialogOptions: "dontDisplay" }
                    }
                ],
                { synchronousExecution: true }
            );

            // 2) 直接置入图片（让 Photoshop 自己按选区/画布处理对齐与尺寸）
            await batchPlay(
                [
                    {
                        _obj: "placeEvent",
                        null: { _path: assetToken, _kind: "local" },
                        freeTransformCenterState: { _enum: "quadCenterState", _value: "QCSAverage" },
                        _options: { dialogOptions: "dontDisplay" }
                    }
                ],
                { synchronousExecution: true }
            );

            // 4) 清除选区（保留你原来的行为：置入前/后都不受选区影响）
            await batchPlay(
                [
                    {
                        _obj: "set",
                        _target: [{ _ref: "channel", _property: "selection" }],
                        to: { _enum: "ordinal", _value: "none" },
                        _options: { dialogOptions: "dontDisplay" }
                    }
                ],
                { synchronousExecution: true }
            );
        }, { commandName: "插入图片到选区" });

    } catch (error) {
        console.log('❌ 添加图像到选区失败:', error.message);
    }
}

// 从Vue端接收图片URL列表并保存到本地文件夹
async function saveAllHistoryImagesFromVue(data) {
    const items = Array.isArray(data?.items) ? data.items : [];
    if (!items.length) {
        console.log('⚠️ saveAllHistoryImagesFromVue: 没有图片可保存');
        return;
    }

    try {
        const fs = require('uxp').storage.localFileSystem;
        const { core } = require('photoshop');

        // 选择保存文件夹（UXP会弹窗）
        const folder = await fs.getFolder();
        if (!folder) {
            console.log('🤷‍ 用户取消了选择文件夹');
            return;
        }

        // 逐个下载并保存（不需要PS模态操作，但这里也不影响）
        let savedCount = 0;
        for (let i = 0; i < items.length; i++) {
            const { imageUrl, filename } = items[i] || {};
            if (!imageUrl) continue;

            const safeName = String(filename || `history_${i + 1}.png`).replace(/[\\/:*?"<>|]/g, '_');

            try {
                console.log(`📥 下载(${i + 1}/${items.length}):`, imageUrl);
                const resp = await fetch(imageUrl);
                if (!resp.ok) throw new Error(`下载失败: ${resp.status} ${resp.statusText}`);

                const arrayBuffer = await resp.arrayBuffer();
                const file = await folder.createFile(safeName, { overwrite: true });
                await file.write(arrayBuffer);

                savedCount++;
                console.log(`✅ 已保存: ${safeName}`);
            } catch (oneErr) {
                console.error(`❌ 保存失败: ${safeName}`, oneErr.message || oneErr);
            }
        }

        console.log(`✅ 批量保存完成: ${savedCount}/${items.length}`);

        // 可选：通知webview完成
        const webview = document.getElementById('panelWebview');
        if (webview) {
            webview.postMessage({
                action: 'saveAllHistoryImagesDone',
                savedCount,
                total: items.length,
                timestamp: new Date().toISOString()
            });
        }

    } catch (error) {
        console.error('💥 批量保存历史图片失败:', error.message || error);
    }
}

// 从Vue端接收数据并保存为本地文本文件
async function saveTextFileFromVue(data) {
    const { filename, content, mimeType } = data;
    if (!filename || !content) {
        console.error('❌ 保存文本文件失败: 文件名或内容为空');
        return;
    }

    try {
        const fs = require('uxp').storage.localFileSystem;
        const { core } = require("photoshop");

        await core.executeAsModal(async () => {
            // 弹出文件保存对话框，可以根据mimeType建议文件类型
            const file = await fs.getFileForSaving(filename, { types: ['json', 'txt', 'log'] });
            if (!file) {
                console.log('🤷‍ 用户取消了文件保存');
                return;
            }

            // 写入文件内容
            await file.write(content, { append: false, format: require('uxp').storage.formats.utf8 });

            console.log(`✅ 文件已保存: ${file.nativePath}`);

        }, { commandName: `保存文件: ${filename}` });

    } catch (error) {
        console.error(`💥 保存文件 ${filename} 失败:`, error.message);
    }
}

// 执行Photoshop动作
async function executeAction(actionData, actionName, timestamp) {
    try {
        console.log('⚡ 开始执行动作:', actionName);
        console.log('📋 原始动作数据:', actionData);
        console.log('📋 数据类型:', typeof actionData);

        let actions;

        // 如果actionData是字符串，尝试解析为JSON
        if (typeof actionData === 'string') {
            console.log('📋 检测到字符串格式，尝试解析JSON...');
            try {
                // 首先尝试直接解析
                actions = JSON.parse(actionData);
                console.log('✅ JSON解析成功:', actions);
            } catch (parseError) {
                console.log('❌ 直接JSON解析失败:', parseError.message);
                console.log('📋 尝试修复JSON格式...');

                try {
                    // 尝试修复JSON格式：为属性名添加引号
                    let fixedJson = actionData
                        .replace(/(\w+):/g, '"$1":')  // 为属性名添加引号
                        .replace(/"(\w+)":/g, '"$1":'); // 确保引号正确

                    console.log('📋 修复后的JSON字符串:', fixedJson);
                    actions = JSON.parse(fixedJson);
                    console.log('✅ 修复后JSON解析成功:', actions);
                } catch (fixError) {
                    console.log('❌ 修复后JSON解析仍然失败:', fixError.message);
                    console.log('📋 尝试使用eval解析...');

                    try {
                        // 最后尝试使用eval（不推荐，但作为后备方案）
                        actions = eval('(' + actionData + ')');
                        console.log('✅ eval解析成功:', actions);
                    } catch (evalError) {
                        console.log('❌ eval解析也失败:', evalError.message);
                        throw new Error('无法解析动作数据，所有解析方法都失败');
                    }
                }
            }
        } else if (Array.isArray(actionData)) {
            // 如果已经是数组，直接使用
            actions = actionData;
            console.log('✅ 使用原始数组数据');
        } else {
            // 如果是单个对象，包装成数组
            actions = [actionData];
            console.log('✅ 包装单个对象为数组');
        }

        // 确保actions是数组
        if (!Array.isArray(actions)) {
            actions = [actions];
        }

        console.log('📋 最终动作数组:', actions);
        console.log('📋 动作数组长度:', actions.length);

        const {batchPlay} = require("photoshop").action;
        const {executeAsModal} = require("photoshop").core;

        await executeAsModal(async () => {
            await batchPlay(actions, {
                synchronousExecution: true,
            });

            console.log('✅ 动作执行完成:', actionName);
        }, {commandName: `执行动作: ${actionName}`});

    } catch (error) {
        console.log('❌ 执行动作失败:', error.message);
        console.log('🔍 动作名称:', actionName);
        console.log('📋 动作数据:', actionData);
        console.log('📋 错误堆栈:', error.stack);
    }
}